*************************************************************************
*                                                                       *
* Program for replication of estimations in Juselius book		*
* (The Cointegrated VAR Model: Econometric Methodology and Macro-	*
* economic Applications)						*
*                                                                       *
* Based on original code by Robert Wagner, October, 15th, 2004 					*
* Revisions by Tom Maycock, Estima, October, 2009
*************************************************************************

*****************************************************************
* Load and transform data					*
*****************************************************************

calendar 1973 1 4
allocate  2003:1
open data book.xls
data(format=xls,org=obs) / lyr lpy lm3n Rm Rb lm3rC dpy

* NOTE!
* Lm3rC used in the analysis is the log of real M3, corrected for the mistake in 1999:1-3
*  and seasonally adjusted.
*
* Rm and Rb are the quaterly interest rates calculated as dividing by 400.
*
* The original book.xls had the series name Dlpy, rather than Dpy, so we
*  renamed the series on the file to Dpy

* Command to compute the logged real M3 from logged nominal M3 and logged price levels:
set lm3r = lm3n-lpy

* We already have DPY from the data file, but it could also be computed using:
* set dpy = lpy - lpy{1}
* or
* diff lpy / dpy


*****************************************************************
* Dummy variables						*
*****************************************************************

set Dt754 = 0.0
compute Dt754(1975:4) = 1
compute Dt754(1976:1) = -0.5
compute Dt754(1976:2) = -0.5

* or:
* set Dt754 = t==1975:4
* set Dt754 1976:1 1976:2 = -0.5


set Dp764 = 0.0
compute Dp764(1976:4) = 1

* or, simply:
* set Dp764 = t==1976:4

set Ds831 = T>=83:1

* Checking the data:
print
table

print(window="Interest Rates") / Rm Rb

graph(header="Interest Rates") 2
# Rm
# Rb

*****************************************************************
* Chapter 2: Detrended series in figure 2.4	           	*
*****************************************************************

set trend = t

* Detrending regression for income::
linreg lyr
# constant trend

* Original code did this:
set tradjlyr = lyr - (0.0037216934*trend + 6.5482173338)

* An easier and more accurate way is to reference %BETA, which is set by LINREG:
* SET tradjlyr = lyr - (%beta(2)*trend + %beta(1))
*
* Or, even easier, just use the residuals parameter on the LINREG:
*
* linreg lyr / tradjlyr
* # constant trend


* Same operations for prices
linreg lpy
# constant trend

* Again, could use %BETA rather than typing in values, but
*  we'll use the hard coded method used to get the results
*  in the text:
set tradjlpy = lpy - (0.012101365*trend - 1.095575207)

* Graphing the trend-adjusted series:
graph(header='Trend adjusted real income')
# tradjlyr

graph(header='Trend adjusted price level')
# tradjlpy

* Both series on a single graph:
graph(header='Trend adjusted income and prices',key=upleft) 2
# tradjlyr
# tradjlpy


* Reproducing Figure 2.4, using SPGRAPH to put 3 graphs on a single page:

spgraph(vfields=3)
 graph(header='Trend adjusted log price: stochastic I(2) trend')
 # tradjlpy
 graph(header='Trend adjusted real aggregate income: stochastic I(1) trend')
 # tradjlyr
 graph(header='Inflation rate: stochastic I(1) trend')
 # dpy
spgraph(done)


*****************************************************************
* Chapter 3 and Chapter 4:                                      *
*****************************************************************

* Note that M3 is the uncorrected series so far.

* Source in the main CATS procedure file:
source c:\cats2\cats.src

* Execute the CATS procedure:
*
* Original code included several "NO" options and an explict date range:
* @cats(noexo,noshift,nodum,lags=2,season=4,dettrend=drift) 1973:1 2003:1
* # lm3r lyr dpy rm rb
*
* However, those are the defaults for those options, and the default date range, so we can just do:

@cats(lags=2,season=4,dettrend=drift)
# lm3r lyr dpy rm rb

* Using CATS:
*  Figures 3.3, 3.4:  Use the Graphics-Series operation in CATS
*  Figure 3.5:        Use the Graphics-Residuals operation
*  Table 3.1,Fig 3.6: Use the Graphics-Roots of the Companion Matrix
*                      operation (selecting rank 5)

*
* Table 4.1 (and F-tests on preceeding page)
* Estimating Unrestricted VAR in RATS:
*
*  Exit out of CATS, and do the following:
*

* Define centered seasonal dummy:
seasonal(centered) Dq

* Define the VAR model:
system(model=var2mod)
variables lm3r lyr dpy rm rb
lags 1 2
deter dq{0 1 2} constant
end(system)

* Estimate the model:
estimate

display 'Residual covariance matrix = ' ##.#### %sigma
display 'Residual correlation matrix = ' ##.#### %cvtocorr(%sigma)
display 'Log likelihood = ' %logl
display 'Log determinant = ' %logdet

* Cross equation F-statistics. Result in book is off by factor of 105/101 (apparently
* due to double-counting the deterministic coefficients)
*
compute eqn=%modeleqn(var2mod,1)
compute betaols=%modelgetcoeffs(var2mod)

* Use REPORT feature to generate the output table:
report(action=define)
report(atrow=1,atcol=1) "Label" "F-Stat"
do i=1,%eqnsize(eqn)
   compute fstat=%qform(inv(%sigma),%xrow(betaols,i))*(%nobs-%nreg)/(5*%nobs*%xx(i,i))
   report(row=new,atcol=1) %eqnreglabels(eqn)(i) fstat
   if %ftest(fstat,%nvar,%nobs-%nreg)<0.05
      report(action=format,atrow=%reportrow,atcol=2,special=onestar)
end do i
report(action=format,atcol=2,picture="*.##",align=decimal)
report(action=show)
*
compute xx10=inv(%xsubmat(%xx,1,10,1,10))
compute beta10=%xsubmat(betaols,1,10,1,5)
compute test=tr(beta10)*xx10*beta10
display "F-test on all regressors" %dot(test,inv(%sigma))*(%nobs-%nreg)/(50.0*%nobs)

* Table 4.2:
* VECM model, m=1
diff lm3r / dlm3r
diff lyr / dlyr
diff dpy / ddpy
diff rm / drm
diff rb / drb

system(model=vecm1mod)
variables dlm3r dlyr ddpy drm drb
lags 1
deter dq{0 1 2} lm3r{1} lyr{1} dpy{1} rm{1} rb{1} constant
end(system)

* Estimate the model:
estimate

display 'Log likelihood = ' %logl
display 'Log determinant = ' %logdet

* Table 4.3:
* VECM model, m=2

system(model=vecm1mod)
variables dlm3r dlyr ddpy drm drb
lags 1
deter dq{0 1 2} lm3r{2} lyr{2} dpy{2} rm{2} rb{2} constant
end(system)

* Estimate the model:
estimate

display 'Log likelihood = ' %logl
display 'Log determinant = ' %logdet

* Table 4.4 is similar.

* Now we'll reproduce Table 4.2, and other results, using CATS:

* First, load CATS again:
@cats(lags=2,season=4,dettrend=drift)
# lm3r lyr dpy rm rb

* Table 4.2: Use Misc-Short Run Parameters to display the parmeters on the
*             lagged differences.

* Figures 4.1-4.5:
*  Use Graphics-Residuals. Select (highlight) all 5 series and then
*   click on OK.

* Figures 4.6:
*  Use Graphics-Correlations. Select (highlight) all 5 series
*   and click OK.

* Table 4.5:
*  Do Misc-Lag Length Determination. Use default setting of 5 lags.

* Various specification tests, pages 73-77
*  Use operation Misc-Residual Analysis to produce these


***************************************************************
* Chapter 5	                                                    *
*****************************************************************
* For the Pi matrix, see the initial output from executing CATS above.
*

* Figure 5.1:
* Execute out of CATS and do the following:

* Simple regression:
linreg lyr
# constant lm3rc

* Compute fitted values:
prj fitted

* Draw the graph. Using OVERLAY and OVSAME to draw the fitted values as a line, while the
*  main cross plot is done using DOTS:

scatter(style=dots,overlay=line,ovsame,vlabel='Real Income',hlabel='Real Money Stock',frame=half) 2
# lyr lm3rc
# fitted lm3rc / 1

***************************************************************
* Chapter 6	                                                    *
*****************************************************************

* Note that we will now be using the corrected M3, which fixed an error in 1999
* and omits seasonal effects.

* To reproduce Figure 6.3, we need to compute the nominal corrected M3 from LM3RC:
set lm3nc = lm3rc + lpy

* Graphing Figure 6.3
graph(footer="The original and corrected M3 in logs.",key=upleft,$
      max=6.5,pattern) 2
# lm3nc
# lm3n

* This graph shows a shorter subsample, making the de-seasonalization
*  more obvious
graph(footer="The original and corrected M3 in logs.",key=upleft,$
      max=6.5,pattern) 2
# lm3nc 1985:1 1992:1
# lm3n 1985:1 1992:1


* Now, exit CATS and restart, with some changes:
* 1) Remove the seasonal dummy term, by deleting the seasonal=4 option
* 2) Add Ds831 as a weakly exogenous variable, via the EXO option and
*    the second supplementary card.
* 3) Include Dt754 and Dp764 as dummy variables, via the DUM option and
*    the third supplementary card.

@cats(lags=2,exo,dum,dettrend=cidrift)
# lm3rc lyr dpy Rm Rb
# Ds831
# Dt754 Dp764

* Use the following menu operations:
*  The middle portion of Table 6.3 is given by the Pi matrix in the initial output.
*  Misc-Residual Analysis to produce Table 6.2 and the residual covariance matrix of Table 6.3
*       Note: CATS now does LM(1) and LM(2) tests for autocorrelation,
*             rather than LM(1) and LM(4) as shown in the text.
*  Graphics-Roots of Companion Matrix, rank 5, for bottom of Table 6.3
*  Misc-Short Run Parameters to produce the first and third portions
*   of the coefficient table in Table 6.3 (note that third coefficient array in
*   the text is transposed)

*****************************************************************
* Chapter 7, 8 and 9                                            *
*****************************************************************

* Note that we now use the 'shift' option for Ds831 instead of the 'exo'
* because we exlude DDs831(t-1) and only include DDs831(t)

* Exit out of CATS, and restart it using these options:
*
@cats(lags=2,shift,dum,dettrend=cidrift)
# lm3rc lyr dpy Rm Rb
# Ds831
# Dt754 Dp764

* Table 7.1:
*  The non-normalized eigenvectors and PI matrix are shown automatically.
*
*  To get the normalized eigenvectors and Alpha matrix, select
*    "Set Rank of Pi" from the "I(1)" menu and enter 5 as the rank.
*
*  Then normalize as indicated on the text (row 1 on Mr (DPY), row 2
*    on Yr (LM3RC), and so on.

* Figures 7.1-7.5:
*  Select "Cointegrating Relations" from the "Graphics" menu, highlight
*   all 5 series, and click OK.

* Table 8.1:
*  Select "Rank Test Statistics" from the "I(1)" menu for the first
*   half of the table.
*
*  Select "Roots of Companion Matrix" from the "Graphics" menu to get
*   the second part of the table.

* Figures 9.1-9.12:
*  Select "Set Rank of Pi" from the "I(1)" menu and enter 3 as the rank.
*   Accept the suggested normalization.
*
*  Then, select "Recursive Estimation" from the "I(1)" menu. Change
*   the ending date of the Base Sample to 1984:4 (as noted on p. 151).
*
*  Use the "Select All" button to get all the graphs. Say "no" when
*   asked about saving the residuals, unless you wish to save them at this
*   time.
*
*  Figure 9.8 is produced using a different estimation sample. To produce
*   this, do "Recursive Estimation" again, answer "Yes" to re-estimate,
*   set the starting date of the "Known Beta" range to 1986:1 and click
*   OK. Select the checkbox for the "Test of Beta(t) = Known Beta" graph
*   and click OK.
*
*  Figure 9.9 is produced using yet another sample. To produce
*   this, do "Recursive Estimation" again, answer "Yes" to re-estimate,
*   reset the starting date of the "Known Beta" range back to 1973:3 and
*   and set the ending date of the "Known Beta" range to 1986:1. Click
*   OK. Select the checkbox for the "Test of Beta(t) = Known Beta" graph
*   and click OK.
*
* Backwards recursive tests -- Figures 9.13-9.21
*   Procedure is basically the same as above, except using "Backwards
*     Recursive Estimation" from the "I(1)" menu.
*
*   Set the starting date for the Base Sample to 1986:1.
*
*   Figure 9.18 is done using a different estimation range. Do "Backwards
*     Recursive Estimation" and set the "Known Beta" range to run from 1983:1
*     through 2003:1. The start of the "Base Sample" remains 1986:1. After
*     estimating, select "Test of Beta(t) = Known Beta" to generate the graph.

*****************************************************************
* Chapter 10 - Restrictions on beta		                         *
* - First a model with a restricted trend (as in chp.8)         *
* - Then the model is respecified with an unrestricted const.   *
*****************************************************************

* Restricted trend

* If you have exited CATS, restart using this model:

@cats(lags=2,shift,dum,dettrend=cidrift)
# lm3rc lyr dpy Rm Rb
# Ds831
# Dt754 Dp764

* Set the Rank to 3
*
* Normalize as indicated in the text (row 1 on DPY, row 2 on LM3RC,
*  and row 3 on RM).
*
* Tables 10.1 and 10.2
*
*  First, we'll impose the H1 hypothesis (Example 1 on page 178 of the text)
*     using the Beta = H*Phi formulation.
*
* H1: Do the following:
*  1) Select "Restrictions on Subsets of Beta" from the "I(1)" matrix,
*  2) Accept the defalut values on the first dialog (one subset, with
*      three vectors in that subset),
*  3) Enter 1 as the number of restrictions
*  4) Enter the following H' matrix (note that
*      CATS takes the transpose of the H matrix shown
*      in the text:
*
*     |1,0,0,0,0,0,0|
*     |0,1,0,0,0,0,0|
*     |0,0,1,0,0,0,0|
*     |0,0,0,1,0,0,0|
*     |0,0,0,0,1,0,0|
*     |0,0,0,0,0,1,0|
*
*  5) Accept the suggest degrees of freedom
*  6) Use same normalization as described above.
*
*  Compare the Beta (transposed) output with the "H1" section of Table 10.1
*
*
*  As noted in the text, most of these hypotheses are more easily handled
*     using the "R" formulation, where you specify R such that R'*Beta = 0.
*
*  To use that formulation, select "Change Restriction Formulation" from the
*     "I(1)" menu and choose the "R'*(Beta/Tau) = 0" formulation. Note that
*     you can also use the "Preferences" operation to change the default choice
*     for this setting.
*
* H1: Do the following:
*  1) Select "Restrictions on Subsets of Beta" from the "I(1)" matrix,
*  2) Accept the defalut values on the first dialog (one subset, with
*      three vectors in that subset),
*  3) Enter 1 as the number of restrictions
*  4) Enter the value 1 in the "Trend" column of R, as shown in the text.
*  5) Accept the suggest degrees of freedom
*  6) Use same normalization as described above.
*
* H2: As above, except set the D831 column of R to 1.
*
* H3: As above, except using the R' as shown in the text.
*
* H4: As above, but with 1s in the RM and RB columns of R.
*
* H5: This time, enter 2 as the number of restrictions, and enter R
*      as shown in the text.
*
* H6: This time, enter 3 restrictions, and combine the values for R
*      from H1, H3, and H4. That is, set row 1 of R according to the
*      pattern for H1, row 2 according to the pattern used for H3, and
*      row 3 to the pattern used for H4.
*
*************************************************************************
* Table 10.2: 	Just compare unrestricted pi estimates (r=3)    	*
*		with pi under H6 (in the above output)			*
*************************************************************************

*

* Unrestricted constant

* Exit CATS and restart with this model:

@cats(lags=2,shift,dum,dettrend=drift)
# lm3rc lyr dpy Rm Rb
# Ds831
# Dt754 Dp764

* Set the Rank to 3
* Normalize as indicated in the text (row 1 on DPY, row 2 on LM3RC,
*  and row 3 on RM).

* Table 10.3
* H7-H9: The "Variable Stationarity" operation on the "Automated Tests" menu handles
*          all of these tests automatically. NOTE: Turn OFF the "Shift Dummies" switch
*          in the dialog box to reproduce the results in rows 1 to 3 of Table 10.3 (row r=3, DPY,
*          RM, and RB columns, repsectively, in the CATS output).
*
*        Full results can be obtained using the "Restrictions on Subsets of Beta" operation:
*
*        For this and subsequent tests, we'll demonstrate the "Beta=H*Phi" formulation--use
*          the "Change Restriction Formulation" operation to switch back to that formulation.
*
*        For H7, choose "Restrictions on Subsets of Beta" from the "I(1)" menu.
*        Use the dialog box to change the number of subsets to 2, and to select
*         1 vector in the first subset and 2 vectors in the second subset.
*
*        Enter 5 as the number of restrictions on subset 1, and enter zeros for every column but DPY
*          (which sould be set to 1) in the H' matrix.
*        Enter 1 restriction on subset 2 (one "1" in each column of H', except for DPY, which has zeros in all rows.
*          The H' arrays look like this:
*            H' on subset 1 = [0,0,1,0,0,0]
*            H' on subset 2 =
*                     |1,0,0,0,0,0|
*                     |0,1,0,0,0,0|
*                     |0,0,0,1,0,0|
*                     |0,0,0,0,1,0|
*                     |0,0,0,0,0,1|
*
*            Choosing zero restrictions on subset 2 produces essentially the same results.
*
*        Normalize as indicated in the text (row 1 on DPY, row 2 on LM3RC, and row 3 on RM).

*
* H8 and H9 are similar to above, just restricting a different variable.
*
* H10: Restrictions on Subsets of Beta, 2 subsets (1 and 2 vectors each, respectively).
*            H' on 1   (5 restrictions) = [0,0,1,-1,0,0]
*            H' on 2 = (1 restriction, zeros in column for RM)
*                     |1,0,0,0,0,0|
*                     |0,1,0,0,0,0|
*                     |0,0,1,0,0,0|
*                     |0,0,0,0,1,0|
*                     |0,0,0,0,0,1|
*
*            Normalize vectors 1 and 3 on DPY, vector 2 on LM3RC.
*

* H11: As above, but:
*            H' on 1   (5 restrictions) = [0,0,1,0,-1,0]
*            H' on 2 = (1 restriction, zeros in column for RB)
*                     |1,0,0,0,0,0|
*                     |0,1,0,0,0,0|
*                     |0,0,1,0,0,0|
*                     |0,0,0,1,0,0|
*                     |0,0,0,0,0,1|
*
*            Normalize vector 1 on DPY, vector 2 on LM3RC, vector 3 on RM.

* H12: As above, but:
*            H' on 1   (5 restrictions) = [0,0,0,1,-1,0]
*            H' on 2 = (1 restriction, zeros in column for RB)
*                     |1,0,0,0,0,0|
*                     |0,1,0,0,0,0|
*                     |0,0,1,0,0,0|
*                     |0,0,0,1,0,0|
*                     |0,0,0,0,0,1|
*
*            Normalize vectors 1 and 3 on RM, vector 2 on LM3RC.

* H13: As above, but:
*            H' on 1   (4 restrictions)
*                     [1,-1,0,0,0,0]
*                     [0, 0,0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831)
*                     |1,0,0,0,0,0|
*                     |0,1,0,0,0,0|
*                     |0,0,1,0,0,0|
*                     |0,0,0,1,0,0|
*                     |0,0,0,0,1,0|
*
*            Normalize on LM3RC, DPY, and RM.

* H14: As above, but:
*            H' on 1   (3 restrictions)
*                     [1,-1,0,0,0,0]
*                     [0, 0,1,0,0,0]
*                     [0, 0,0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on LM3RC, DPY, and RM.

* H15: As above, but:
*            H' on 1   (3 restrictions)
*                     [1,-1,0, 0,0,0]
*                     [0, 0,0,-1,1,0]
*                     [0, 0,0, 0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on LM3RC, DPY, and RM.

* H16: As above, but:
*            H' on 1   (2 restrictions)
*                     [1,-1,0, 0,0,0]
*                     [0, 0,1, 0,0,0]
*                     [0, 0,0,-1,1,0]
*                     [0, 0,0, 0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on LM3RC, DPY, and RM.



* H17: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,1,0,0,0,0]
*                     [0,0,1,0,0,0]
*                     [0,0,0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on LYR, DPY, and RM.

* H18: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,1, 0,0,0,0]
*                     [0,0,-1,1,0,0]
*                     [0,0, 0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on LYR, DPY, and RM.

* H19: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,1, 0,0,0,0]
*                     [0,0,-1,0,1,0]
*                     [0,0, 0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on LYR, DPY, and RM.

* H20: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,1,0, 0,0,0]
*                     [0,0,0,-1,1,0]
*                     [0,0,0, 0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on LYR, DPY, and RM.

* H21: As above, but:
*            H' on 1   (4 restrictions)
*                     [0,0,1,0,0,0]
*                     [0,0,0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on DPY, LM3RC, and RM.

* H22: As above, but:
*            H' on 1   (4 restrictions)
*                     [0,0,1,-1,0,0]
*                     [0,0,0, 0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on DPY, LM3RC, and RM.

* H23: As above, but:
*            H' on 1   (4 restrictions)
*                     [0,0,1,0,-1,0]
*                     [0,0,0,0, 0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on DPY, LM3RC, and RM.

* H24: As above, but:
*            H' on 1   (4 restrictions)
*                     [0,0,0,1,-1,0]
*                     [0,0,0,0, 0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on RM, LM3RC, and RM.

* H25: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,0,1,0,0,0]
*                     [0,0,0,1,0,0]
*                     [0,0,0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on DPY, LM3RC, and RM.


* H26: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,0,1,0,0,0]
*                     [0,0,0,0,1,0]
*                     [0,0,0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on DPY, LM3RC, and RM.

* H27: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,0,0,1,0,0]
*                     [0,0,0,0,1,0]
*                     [0,0,0,0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on RM, LM3RC, and RM.

* H28: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,0,1,-1,0,0]
*                     [0,0,0,-1,1,0]
*                     [0,0,0, 0,0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on RM, LM3RC, and RM.

* H29: As above, but:
*            H' on 1   (3 restrictions)
*                     [0,0,1,0,0,0]
*                     [0,0,0,1,-1,0]
*                     [0,0,0,0, 0,1]
*            H' on 2 = (1 restriction, zeros in column for DS831, as above)
*
*            Normalize on RM, LM3RC, and RM.


